% ------------------------------------------------------------------------
% Analysis script for CSQ-SF-D 
% 
% For data and description see: 
% 
% German translation and validation of the Cognitive Style Questionnaire short
% form (CSQ-SF-D)
% Huys et al., 2016, PLoS One 
% 
% Quentin Huys 2016
% http://quentinhuys.com/abstr_HuysEa16-CSQ-SF-D.html

clear all; 

load DataForPublication.mat; 

alldata = Data.alldata; 
age = Data.age; 
sex = Data.sex; 
Qall = Data.Qall; 

valid = ~isnan(sum(alldata,2));
Nsj = sum(valid);

data = alldata(valid,:);
total = sum(data')/360;

tmp = [1 6]'*ones(1,8) + ones(2,1)*(0:9:63); ind{1} = tmp(:);	 % internality 
tmp = [2 7]'*ones(1,8) + ones(2,1)*(0:9:63); ind{2} = tmp(:);	 % globality 
tmp = [3 8]'*ones(1,8) + ones(2,1)*(0:9:63); ind{3} = tmp(:);	 % stability 
tmp = [5 9]'*ones(1,8) + ones(2,1)*(0:9:63); ind{4} = tmp(:);	 % self-worth 
tmp = 4'*ones(1,8) + ones(1,1)*(0:9:63);     ind{5} = tmp(:);	 % negative consequences

subscale = {'Internality','Globality','Stability','SelfWorth','NegCons','Total'};

for k=1:5
	fac(:,k) = sum(data(:,ind{k}),2);
end

fprintf('--------------------------------------------------------------------------------\n')
[c,p] = corr([fac total']);

cp = c; 
cp(p>.05)=NaN;
tab = array2table(round(cp*100)/100);
tab.Properties.VariableNames = subscale; 
tab.Properties.RowNames = subscale;      
tab

% ------------------------------------------------------------------------------------------
% Reliabilities 

cronbach_all = 72/71*(1-sum(var(data))/var(sum(data')));
cronbach_fac = 5/4*(1-sum(var(fac))/var(sum(fac')));
for k=1:5 
	cronbach_sub(k) = sum(ind{k})/(sum(ind{k})-1)*(1-sum(var(data(:,ind{k})))/var(sum(data(:,ind{k}),2)));
end

for k=1:8; 
	scen(:,k) = sum(data(:,(1:8)+(k-1)*8),2);
end
cronbach_scenarios= 8/7*(1-sum(var(scen))/var(sum(scen')));

% ------------------------------------------------------------------------------------------
% Principal component analyses 

subplot(241);
covariance = cov(data);
[vfull,d] = eig(covariance);
d = diag(d);
mybar(1:72,d(end:-1:1)/sum(d)*100,.7);
ylabel('% Variance explained'); 
xlabel('Principal component'); 
title('Individual items');

subplot(242);
covariance = cov(fac);
[v,d] = eig(covariance);
d = diag(d);
mybar(1:5,d(end:-1:1)/sum(d)*100,.7);
ylabel('% Variance explained'); 
xlabel('Principal component'); 
title('Subscales');

subplot(243);
covariance = cov(scen);
[v,d] = eig(covariance);
d = diag(d);
mybar(1:8,d(end:-1:1)/sum(d)*100,.7);
ylabel('% Variance explained'); 
xlabel('Principal component'); 
title('Scenarios');


% FA on full dataset 
for k=1:2
	[lam,psi,rot,stats,f] = factoran(data,k); 
	bic(k) = -2*stats.loglike + (72*k+72-k*(k-1)/2)*log(Nsj);
end

for k=1:2
	[lam,psi,rot,stats,f] = factoran(fac,k); 
	bic(k) = -2*stats.loglike + (5*k+5-k*(k-1)/2)*log(Nsj);
end

[lam,psi,rot,stats,f] = factoran(data,5); 
for k=1:5
	facload(k,:) = sum(lam(ind{k},:));
end

%-----------------------------------------------------------------------------
% item response analysis  - towards reduced item set 

clear c; 
subplot(2,5,6:8);
for j=1:3 
	if     j==1; ii=1:206; 
	elseif j==2; ii=1:103; 
	elseif j==3; ii=104:206; 
	end

	total = sum(data(ii,:),2); 

	clear itemcorr 
	for k=1:72
		X = [ones(length(ii),1) data(ii,k)];
		itemcorr(:,k) = inv(X'*X)*X'*total;
	end

	[foo,itemorder(:,j)] = sort(-itemcorr(2,:)); % largest effect first - note negative sign! 

	for k=1:72
		c(k,j) = corr(sum(data(ii,itemorder(1:k,j)),2),sum(data(ii,:),2));
	end
end

plot(c); 
hon
plot(c(:,1),'linewidth',3); 
plot([0 73],[.95 .95],'k');
hof
ylim([0.5 1]);
xlim([0 73]);
grid on
set(gca,'xtick',[1:72],'xticklabel',itemorder(:,1),'xticklabelrotation',90)
title('Individual items');
ylabel('Correlation with total score');

for k=1:72;
	itemrelativeranking(k,1) = find(itemorder(:,2)==itemorder(k,1));
	itemrelativeranking(k,2) = find(itemorder(:,3)==itemorder(k,1));
	itemrelativeranking(k,3) = find(itemorder(:,3)==itemorder(k,2));
end

bestitems_item = mod(itemorder(1:27,1),9); 
bestitems_item(bestitems_item==0) = 9; 
bestitems_scenario= ceil(itemorder(1:27,1)/9); 
clear n 
for k=1:5
	item = mod(intersect(itemorder(1:27,1),ind{k}),9); 
	item(item==0) = 9; 
	scen2 = ceil(intersect(itemorder(1:27,1),ind{k})/9);
	bestitems_countsubscores(k) = length(intersect(itemorder(1:27,1),ind{k}));
end

% Print out latex table of ranked items 
fprintf('--------------------------------------------------------------------------------\n')
fprintf('Rank')
fprintf('  %i',1:27)
fprintf('\n')
fprintf('Scenario')
fprintf('  %i',bestitems_scenario')
fprintf('\n')
fprintf('Item')
fprintf('  %i',bestitems_item')
fprintf('\n')
fprintf('--------------------------------------------------------------------------------\n')

[foo,i] = sort(bestitems_scenario); 
[bestitems_scenario(i) bestitems_item(i)]; 

clear c p
for k=2:5
	i = intersect(itemorder(1:27,1),ind{k});
	[c(k),p(k)] = corr(sum(data(:,itemorder(i,1)),2),fac(:,k));
end

% PCA on reduced item set 

subplot(244);
covariance = cov(data(:,itemorder(1:27,1)));
[vsub,d] = eig(covariance);
d = diag(d);
mybar(d(end:-1:1)/sum(d)*100,.7);
ylabel('% Variance explained')
xlabel('Principal component')
title('Reduced item set');

[c,p] = corr(data(:,itemorder(1:27,1))*vsub(:,end),data*vfull(:,end)); 

cronbach_veryshort= 27/26*(1-sum(var(data(:,itemorder(1:27,1))))/var(sum(data(:,itemorder(1:27,1))')));

% factor response analysis 

total = sum(data,2)/360;
for k=1:5 
	X = [ones(206,1) fac(:,k)];
	faccorr(:,k) = inv(X'*X)*X'*total; 
end
[foo,facorder] = sort(-faccorr(2,:)); % largest effect first - note negative sign! 

subplot(259);
clear c; 
for k=1:5
	c(k) = corr(sum(fac(:,facorder(1:k)),2),total);
end
plot(c,'-o'); 
ylim([0.5 1]);
xlim([0 6]);
hon
plot([0 6],[.95 .95],'k');
hof
set(gca,'xtick',[1:5],'xticklabel',subscale(facorder),'xticklabelrotation',40)
grid on
title('Subscales');

% scenario response analysis 
total = sum(data,2);
for k=1:8 
	X = [ones(206,1) scen(:,k)];
	scenariocorr(:,k) = inv(X'*X)*X'*total; 
end
[foo,i] = sort(-scenariocorr(2,:)); % largest effect first - note negative sign! 

subplot(2,5,10);
clear c; 
ii = []; 
for k=1:8
	ii = [ii (1:8)+(i(k)-1)*8];
	c(k) = corr(sum(data(:,ii),2),total);
end
plot(c,'-o')
grid on
ylim([0.5 1]);
xlim([0 9]);
hon
plot([0 9],[.95 .95],'k');
hof
set(gca,'xtick',[1:8],'xticklabel',i)
title('Scenarios');
xlabel('Scenario number');

subplot(2,5,6:8); 
ax = axes('position',[.3,.17 .26 .19]);
plot(1:72,1:72,'k');
hon
plot(itemrelativeranking(:,3));
hof
xlabel('First half rank'); 
ylabel('Second half rank'); 
axis tight


% ---------------------------------------------------------------
% external validity - print latex tables 

clear c p 
for k=1:5
	[c(k),p(k)] = corr(Qall.IDS.Score',sum(Qall.CSQ.Array(:,ind{k}),2));
end
[c(6),p(6)] = corr(Qall.IDS.Score', Qall.CSQ.Score'); 

tab2 = array2table([round(c'*100)/100 p']);
tab2.Properties.VariableNames = {'corr','p'};
tab2.Properties.RowNames = subscale;
tab2

fprintf('--------------------------------------------------------------------------------\n')
fprintf('%s  %s  %s  %s  %s  %s \n',subscale{1},subscale{2},subscale{3},subscale{4},subscale{5},subscale{6});
fprintf('Correlation  %.2g  %.2g  %.2g  %.2g  %.2g  %.2g\n',c);
fprintf('p-value      %.5g  %.5g  %.5g  %.5g  %.2g  %.5g\n',p);
fprintf('\n')

clear c p 
for k=1:5
	[c(k),p(k)] = corr(Qall.STAI.StaitScore',sum(Qall.CSQ.Array(:,ind{k}),2));
end
[c(6),p(6)] = corr(Qall.STAI.StaitScore', Qall.CSQ.Score'); 

fprintf('%s  %s  %s  %s  %s  %s \n',subscale{1},subscale{2},subscale{3},subscale{4},subscale{5},subscale{6});
fprintf('Correlation  %.2g  %.2g  %.2g  %.2g  %.2g  %.2g\n',c);
fprintf('p-value      %.5g  %.5g  %.5g  %.5g  %.2g  %.5g\n',p);
fprintf('\n')

clear c p 
for k=1:5
	[c(k),p(k)] = corr(Qall.STAI.TraitScore',sum(Qall.CSQ.Array(:,ind{k}),2));
end
[c(6),p(6)] = corr(Qall.STAI.TraitScore', Qall.CSQ.Score'); 

fprintf('%s  %s  %s  %s  %s  %s \n',subscale{1},subscale{2},subscale{3},subscale{4},subscale{5},subscale{6});
fprintf('Correlation  %.2g  %.2g  %.2g  %.2g  %.2g  %.2g\n',c);
fprintf('p-value      %.5g  %.5g  %.5g  %.5g  %.2g  %.5g\n',p);
fprintf('\n')

clear c p 
for k=1:5
	[c(k),p(k)] = corr(Qall.SPSRQ.SR',sum(Qall.CSQ.Array(:,ind{k}),2));
end
[c(6),p(6)] = corr(Qall.SPSRQ.SR', Qall.CSQ.Score'); 

fprintf('%s  %s  %s  %s  %s  %s \n',subscale{1},subscale{2},subscale{3},subscale{4},subscale{5},subscale{6});
fprintf('Reward Sensitivity  Linear Correlation  %.2g  %.2g  %.2g  %.2g  %.2g  %.2g\n',c);
fprintf('                    p-value      %.5g  %.5g  %.5g  %.5g  %.2g  %.5g\n',p);
fprintf('\n')

clear c p 
for k=1:5
	[c(k),p(k)] = corr(Qall.SPSRQ.SP',sum(Qall.CSQ.Array(:,ind{k}),2));
end
[c(6),p(6)] = corr(Qall.SPSRQ.SP', Qall.CSQ.Score'); 

fprintf('%s  %s  %s  %s  %s  %s \n',subscale{1},subscale{2},subscale{3},subscale{4},subscale{5},subscale{6});
fprintf('Punishment Sensitivity  Linear Correlation  %.2g  %.2g  %.2g  %.2g  %.2g  %.2g\n',c);
fprintf('                        p-value      %.5g  %.5g  %.5g  %.5g  %.2g  %.5g\n',p);
fprintf('--------------------------------------------------------------------------------\n')


